//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2008 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <richedit.h>
#include <malloc.h>
#include <stdio.h>
#include <hash_map>
#include <vd2/Dita/services.h>
#include <vd2/system/error.h>
#include <vd2/system/file.h>
#include <vd2/system/filewatcher.h>
#include <vd2/system/thread.h>
#include <vd2/system/refcount.h>
#include <vd2/system/vdstl.h>
#include <vd2/system/w32assist.h>
#include "console.h"
#include "ui.h"
#include "uiframe.h"
#include "texteditor.h"
#include "simulator.h"
#include "debugger.h"
#include "resource.h"
#include "disasm.h"
#include "symbols.h"

extern HINSTANCE g_hInst;
extern HWND g_hwnd;

extern ATSimulator g_sim;

void ATConsoleExecuteCommand(char *s);

///////////////////////////////////////////////////////////////////////////

class ATUIPane;

namespace {
	HFONT	g_monoFont;
	HMENU	g_hmenuSrcContext;
}

///////////////////////////////////////////////////////////////////////////

class ATDisassemblyWindow : public ATUIPane,
							public IATDebuggerClient,
							public IVDTextEditorCallback,
							public IVDTextEditorColorizer
{
public:
	ATDisassemblyWindow();
	~ATDisassemblyWindow();

	void OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state);

	void OnTextEditorUpdated();
	void RecolorLine(int line, const char *text, int length, IVDTextEditorColorization *colorization);

	void SetPosition(uint16 addr);

protected:
	VDGUIHandle Create(uint32 exStyle, uint32 style, int x, int y, int cx, int cy, VDGUIHandle parent, int id);
	LRESULT WndProc(UINT msg, WPARAM wParam, LPARAM lParam);

	bool OnCreate();
	void OnDestroy();
	void OnSize();
	void OnSetFocus();
	void RemakeView(uint16 focusAddr);

	vdrefptr<IVDTextEditor> mpTextEditor;
	HWND	mhwndTextEditor;
	HWND	mhwndAddress;
	VDStringA	mState;

	uint32	mViewStart;
	uint32	mViewLength;
	int		mHighlightedLine;
	uint32	mHighlightedPC;
};

ATDisassemblyWindow::ATDisassemblyWindow()
	: ATUIPane(kATUIPaneId_Disassembly, "Disassembly")
	, mViewStart(0)
	, mViewLength(0)
	, mHighlightedLine(-1)
	, mHighlightedPC(0)
{
	mPreferredDockCode = kATContainerDockRight;
}

ATDisassemblyWindow::~ATDisassemblyWindow() {
}

LRESULT ATDisassemblyWindow::WndProc(UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
		case WM_SIZE:
			OnSize();
			break;
	}

	return ATUIPane::WndProc(msg, wParam, lParam);
}

bool ATDisassemblyWindow::OnCreate() {
	if (!ATUIPane::OnCreate())
		return false;

	if (!VDCreateTextEditor(~mpTextEditor))
		return false;

	mhwndAddress = CreateWindowEx(0, WC_COMBOBOXEX, "", WS_VISIBLE|WS_CHILD|WS_CLIPSIBLINGS|CBS_DROPDOWN|CBS_HASSTRINGS|CBS_AUTOHSCROLL, 0, 0, 0, 0, mhwnd, (HMENU)101, g_hInst, NULL);

	mhwndTextEditor = (HWND)mpTextEditor->Create(WS_EX_NOPARENTNOTIFY, WS_CHILD|WS_VISIBLE, 0, 0, 0, 0, (VDGUIHandle)mhwnd, 100);
	SendMessage(mhwndTextEditor, WM_SETFONT, (WPARAM)g_monoFont, NULL);

	mpTextEditor->SetCallback(this);
	mpTextEditor->SetColorizer(this);
	mpTextEditor->SetReadOnly(true);

	OnSize();
	ATGetDebugger()->AddClient(this, true);
	return true;
}

void ATDisassemblyWindow::OnDestroy() {
	ATGetDebugger()->RemoveClient(this);
	ATUIPane::OnDestroy();
}

void ATDisassemblyWindow::OnSize() {
	RECT r;
	if (GetClientRect(mhwnd, &r)) {
		RECT rAddr;
		GetWindowRect(mhwndAddress, &rAddr);

		int comboHt = rAddr.bottom - rAddr.top;

		SetWindowPos(mhwndAddress, NULL, 0, 0, r.right, comboHt, SWP_NOMOVE|SWP_NOZORDER|SWP_NOACTIVATE);
		SetWindowPos(mhwndTextEditor, NULL, 0, comboHt, r.right, r.bottom - comboHt, SWP_NOZORDER|SWP_NOACTIVATE);
	}
}

void ATDisassemblyWindow::OnSetFocus() {
	::SetFocus(mhwndTextEditor);
}

void ATDisassemblyWindow::RemakeView(uint16 focusAddr) {
	uint16 pc = mViewStart;
	char buf[256];

	mpTextEditor->Clear();

	mHighlightedLine = -1;
	int line = 0;
	int focusLine = -1;
	while((uint16)(pc - mViewStart) < 0x100) {
		if (pc == mHighlightedPC)
			mHighlightedLine = line;

		if (pc <= focusAddr)
			focusLine = line;

		++line;

		pc = ATDisassembleInsn(buf, pc);

		mpTextEditor->Append(buf);
	}

	if (focusLine >= 0) {
		mpTextEditor->CenterViewOnLine(focusLine);
		mpTextEditor->SetCursorPos(focusLine, 0);
	}

	mViewLength = pc - mViewStart;
	mpTextEditor->RecolorAll();
}

void ATDisassemblyWindow::OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state) {
	mHighlightedPC = state.mPC;
	mHighlightedLine = -1;

	if (state.mPC - mViewStart >= mViewLength) {
		SetPosition(state.mPC);
	} else {
		uint32 offset = 0;
		int line = 0;
		while(offset < mViewLength) {
			int len = ATGetOpcodeLength(g_sim.DebugReadByte(mViewStart + offset));

			if (mHighlightedPC == (uint16)(mViewStart + offset))
				mHighlightedLine = line;

			offset += len;
			++line;
		}

		if (mHighlightedLine >= 0)
			mpTextEditor->MakeLineVisible(mHighlightedLine);
		mpTextEditor->RecolorAll();
	}
}

void ATDisassemblyWindow::OnTextEditorUpdated() {
}

void ATDisassemblyWindow::RecolorLine(int line, const char *text, int length, IVDTextEditorColorization *cl) {
	if (line == mHighlightedLine) {
		cl->AddTextColorPoint(0, 0x000000, 0xFFFF80);
		return;
	}
}

void ATDisassemblyWindow::SetPosition(uint16 addr) {
	VDSetWindowTextFW32(mhwndAddress, L"$%04X", addr);
	mViewStart = ATDisassembleGetFirstAnchor(addr >= 0x80 ? addr - 0x80 : 0, addr);

	RemakeView(addr);
}

///////////////////////////////////////////////////////////////////////////

class ATRegistersWindow : public ATUIPane, public IATDebuggerClient {
public:
	ATRegistersWindow();
	~ATRegistersWindow();

	void OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state);

protected:
	VDGUIHandle Create(uint32 exStyle, uint32 style, int x, int y, int cx, int cy, VDGUIHandle parent, int id);
	LRESULT WndProc(UINT msg, WPARAM wParam, LPARAM lParam);

	bool OnCreate();
	void OnDestroy();
	void OnSize();

	HWND	mhwndEdit;
	VDStringA	mState;
};

ATRegistersWindow::ATRegistersWindow()
	: ATUIPane(kATUIPaneId_Registers, "Registers")
{
	mPreferredDockCode = kATContainerDockRight;
}

ATRegistersWindow::~ATRegistersWindow() {
}

LRESULT ATRegistersWindow::WndProc(UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
		case WM_SIZE:
			OnSize();
			break;
	}

	return ATUIPane::WndProc(msg, wParam, lParam);
}

bool ATRegistersWindow::OnCreate() {
	if (!ATUIPane::OnCreate())
		return false;

	mhwndEdit = CreateWindowEx(0, "EDIT", "", WS_CHILD|WS_VISIBLE|ES_AUTOHSCROLL|ES_AUTOVSCROLL|ES_MULTILINE, 0, 0, 0, 0, mhwnd, (HMENU)100, VDGetLocalModuleHandleW32(), NULL);
	if (!mhwndEdit)
		return false;

	SendMessage(mhwndEdit, WM_SETFONT, (WPARAM)g_monoFont, TRUE);

	OnSize();

	ATGetDebugger()->AddClient(this);
	return true;
}

void ATRegistersWindow::OnDestroy() {
	ATGetDebugger()->RemoveClient(this);
	ATUIPane::OnDestroy();
}

void ATRegistersWindow::OnSize() {
	RECT r;
	if (GetClientRect(mhwnd, &r))
		SetWindowPos(mhwndEdit, NULL, 0, 0, r.right, r.bottom, SWP_NOZORDER|SWP_NOACTIVATE|SWP_NOMOVE);
}

void ATRegistersWindow::OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state) {
	mState.clear();
	mState.append_sprintf("PC = %04X\r\n", state.mPC);
	mState.append_sprintf("A = %02X\r\n", state.mA);
	mState.append_sprintf("X = %02X\r\n", state.mX);
	mState.append_sprintf("Y = %02X\r\n", state.mY);
	mState.append_sprintf("S = %02X\r\n", state.mS);
	mState.append_sprintf("P = %02X\r\n", state.mP);
	mState.append_sprintf("    %c%c1%c%c%c%c%c\r\n"
		, state.mP & 0x80 ? 'N' : '-'
		, state.mP & 0x40 ? 'V' : '-'
		, state.mP & 0x10 ? 'B' : '-'
		, state.mP & 0x08 ? 'D' : '-'
		, state.mP & 0x04 ? 'I' : '-'
		, state.mP & 0x02 ? 'Z' : '-'
		, state.mP & 0x01 ? 'C' : '-'
		);

	SetWindowText(mhwndEdit, mState.c_str());
}

///////////////////////////////////////////////////////////////////////////

class ATCallStackWindow : public ATUIPane, public IATDebuggerClient {
public:
	ATCallStackWindow();
	~ATCallStackWindow();

	void OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state);

protected:
	VDGUIHandle Create(uint32 exStyle, uint32 style, int x, int y, int cx, int cy, VDGUIHandle parent, int id);
	LRESULT WndProc(UINT msg, WPARAM wParam, LPARAM lParam);

	bool OnCreate();
	void OnDestroy();
	void OnSize();

	HWND	mhwndList;
	VDStringA	mState;
};

ATCallStackWindow::ATCallStackWindow()
	: ATUIPane(kATUIPaneId_CallStack, "Call Stack")
{
	mPreferredDockCode = kATContainerDockRight;
}

ATCallStackWindow::~ATCallStackWindow() {
}

LRESULT ATCallStackWindow::WndProc(UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
		case WM_SIZE:
			OnSize();
			break;
	}

	return ATUIPane::WndProc(msg, wParam, lParam);
}

bool ATCallStackWindow::OnCreate() {
	if (!ATUIPane::OnCreate())
		return false;

	mhwndList = CreateWindowEx(0, "LISTBOX", "", WS_CHILD|WS_VISIBLE|LBS_HASSTRINGS, 0, 0, 0, 0, mhwnd, (HMENU)100, VDGetLocalModuleHandleW32(), NULL);
	if (!mhwndList)
		return false;

	SendMessage(mhwndList, WM_SETFONT, (WPARAM)g_monoFont, TRUE);

	OnSize();

	ATGetDebugger()->AddClient(this);
	return true;
}

void ATCallStackWindow::OnDestroy() {
	ATGetDebugger()->RemoveClient(this);
	ATUIPane::OnDestroy();
}

void ATCallStackWindow::OnSize() {
	RECT r;
	if (GetClientRect(mhwnd, &r))
		SetWindowPos(mhwndList, NULL, 0, 0, r.right, r.bottom, SWP_NOZORDER|SWP_NOACTIVATE|SWP_NOMOVE);
}

void ATCallStackWindow::OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state) {
	IATDebugger *db = ATGetDebugger();
	IATDebuggerSymbolLookup *dbs = ATGetDebuggerSymbolLookup();
	ATCallStackFrame frames[16];
	uint32 n = db->GetCallStack(frames, 16);

	SendMessage(mhwndList, LB_RESETCONTENT, 0, 0);
	for(uint32 i=0; i<n; ++i) {
		const ATCallStackFrame& fr = frames[i];
		ATSymbol sym;
		const char *symname = "";
		if (dbs->LookupSymbol(fr.mPC, kATSymbol_Execute, sym))
			symname = sym.mpName;

		mState.sprintf("%04X: %c%04X (%s)", 0x0100 + fr.mS, fr.mP & 0x04 ? '*' : ' ', fr.mPC, symname);
		SendMessage(mhwndList, LB_ADDSTRING, 0, (LPARAM)mState.c_str());
	}
}

///////////////////////////////////////////////////////////////////////////

class ATSourceWindow : public VDShaderEditorBaseWindow
					 , public IATSimulatorCallback
					 , public IATDebuggerClient
					 , public IVDTextEditorCallback
					 , public IVDTextEditorColorizer
					 , public IVDFileWatcherCallback
{
public:
	ATSourceWindow();
	~ATSourceWindow();

	VDGUIHandle Create(uint32 exStyle, uint32 style, int x, int y, int cx, int cy, VDGUIHandle parent, int id);

	void LoadFile(const wchar_t *s);

protected:
	LRESULT WndProc(UINT msg, WPARAM wParam, LPARAM lParam);

	bool OnCreate();
	void OnDestroy();
	void OnSize();
	bool OnCommand(UINT cmd);

	void OnTextEditorUpdated();
	void RecolorLine(int line, const char *text, int length, IVDTextEditorColorization *colorization);

	void OnSimulatorEvent(ATSimulatorEvent ev);
	void OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state);
	bool OnFileUpdated(const wchar_t *path);

	void	HighlightLine(int line);

	sint32	GetCurrentLineAddress() const;

	uint32	mModuleId;
	uint16	mFileId;

	int		mHighlightedLine;

	vdrefptr<IVDTextEditor>	mpTextEditor;
	HWND	mhwndTextEditor;
	HACCEL	mhAccel;
	VDStringW	mPath;

	typedef stdext::hash_map<uint32, uint32> AddressLookup;
	AddressLookup	mAddressToLineLookup;
	AddressLookup	mLineToAddressLookup;

	VDFileWatcher	mFileWatcher;
};

ATSourceWindow::ATSourceWindow()
	: mHighlightedLine(-1)
	, mhAccel(LoadAccelerators(g_hInst, MAKEINTRESOURCE(IDR_DEBUGGER_ACCEL)))
{
}

ATSourceWindow::~ATSourceWindow() {
}

VDGUIHandle ATSourceWindow::Create(uint32 exStyle, uint32 style, int x, int y, int cx, int cy, VDGUIHandle parent, int id) {
	return (VDGUIHandle)CreateWindowEx(exStyle, (LPCSTR)sWndClass, "", style, x, y, cx, cy, (HWND)parent, (HMENU)id, VDGetLocalModuleHandleW32(), static_cast<VDShaderEditorBaseWindow *>(this));
}

void ATSourceWindow::LoadFile(const wchar_t *s) {
	VDTextInputFile ifile(s);

	mpTextEditor->Clear();

	uint32 lineno = 0;
	bool listingMode = false;
	while(const char *line = ifile.GetNextLine()) {
		if (listingMode) {
			char space0;
			int origline;
			int address;
			char dummy;
			char space1;
			char space2;
			char space3;
			char space4;
			int op;

			bool valid = false;
			if (7 == sscanf(line, "%c%d %4x%c%c%2x%c", &space0, &origline, &address, &space1, &space2, &op, &space3)
				&& space0 == ' '
				&& space1 == ' '
				&& space2 == ' '
				&& space3 == ' ')
			{
				valid = true;
			} else if (8 == sscanf(line, "%6x%c%c%c%c%c%2x%c", &address, &space0, &space1, &dummy, &space2, &space3, &op, &space4)
				&& space0 == ' '
				&& space1 == ' '
				&& space2 == ' '
				&& space3 == ' '
				&& space4 == ' '
				&& isdigit((unsigned char)dummy))
			{
				valid = true;
			} else if (6 == sscanf(line, "%6d%c%4x%c%2x%c", &origline, &space0, &address, &space1, &op, &space2)
				&& space0 == ' '
				&& space1 == ' '
				&& (space2 == ' ' || space2 == '\t'))
			{
				valid = true;
			}

			if (valid) {
				mAddressToLineLookup.insert(AddressLookup::value_type(address, lineno));
				mLineToAddressLookup.insert(AddressLookup::value_type(lineno, address));
			}
		} else {
			if (!lineno && !strncmp(line, "mads ", 5))
				listingMode = true;
		}

		mpTextEditor->Append(line);
		mpTextEditor->Append("\n");
		++lineno;
	}

	if (ATGetDebuggerSymbolLookup()->LookupFile(s, mModuleId, mFileId)) {
		vdfastvector<ATSourceLineInfo> lines;

		ATGetDebuggerSymbolLookup()->GetLinesForFile(mModuleId, mFileId, lines);

		vdfastvector<ATSourceLineInfo>::const_iterator it(lines.begin()), itEnd(lines.end());
		for(; it!=itEnd; ++it) {
			const ATSourceLineInfo& linfo = *it;

			mAddressToLineLookup.insert(AddressLookup::value_type(linfo.mOffset, linfo.mLine - 1));
			mLineToAddressLookup.insert(AddressLookup::value_type(linfo.mLine - 1, linfo.mOffset));
		}
	} else {
		mModuleId = 0;
		mFileId = 0;
	}

	mPath = s;
	mFileWatcher.Init(s, this);
}

LRESULT ATSourceWindow::WndProc(UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
		case WM_CREATE:
			return OnCreate() ? 0 : -1;

		case WM_DESTROY:
			OnDestroy();
			break;

		case WM_SIZE:
			OnSize();
			return 0;

		case WM_APP+100:
			{
				MSG& msg = *(MSG *)lParam;

				if (TranslateAccelerator(mhwnd, mhAccel, &msg))
					return TRUE;
			}
			return FALSE;

		case WM_APP+101:
			if (!mPath.empty())
				LoadFile(mPath.c_str());
			return 0;

		case WM_COMMAND:
			if (OnCommand(LOWORD(wParam)))
				return 0;
			break;

		case WM_CONTEXTMENU:
			{
				int x = GET_X_LPARAM(lParam);
				int y = GET_Y_LPARAM(lParam);
				POINT pt = {x, y};

				if (ScreenToClient(mhwndTextEditor, &pt)) {
					mpTextEditor->SetCursorPixelPos(pt.x, pt.y);
					TrackPopupMenu(GetSubMenu(g_hmenuSrcContext, 0), TPM_LEFTALIGN|TPM_TOPALIGN, x, y, 0, mhwnd, NULL);
				}
			}
			return 0;
	}

	return VDShaderEditorBaseWindow::WndProc(msg, wParam, lParam);
}

bool ATSourceWindow::OnCreate() {
	if (!VDCreateTextEditor(~mpTextEditor))
		return false;

	mhwndTextEditor = (HWND)mpTextEditor->Create(WS_EX_NOPARENTNOTIFY, WS_CHILD|WS_VISIBLE, 0, 0, 0, 0, (VDGUIHandle)mhwnd, 100);
	SendMessage(mhwndTextEditor, WM_SETFONT, (WPARAM)g_monoFont, NULL);

	mpTextEditor->SetReadOnly(true);
	mpTextEditor->SetCallback(this);
	mpTextEditor->SetColorizer(this);

	OnSize();
	ATGetDebugger()->AddClient(this);
	g_sim.AddCallback(this);
	return true;
}

void ATSourceWindow::OnDestroy() {
	mFileWatcher.Shutdown();

	g_sim.RemoveCallback(this);
	ATGetDebugger()->RemoveClient(this);
}

void ATSourceWindow::OnSize() {
	RECT r;
	VDVERIFY(GetClientRect(mhwnd, &r));

	SetWindowPos(mhwndTextEditor, NULL, 0, 0, r.right, r.bottom, SWP_NOMOVE|SWP_NOZORDER|SWP_NOACTIVATE);
}

bool ATSourceWindow::OnCommand(UINT cmd) {
	switch(cmd) {
		case ID_DEBUG_BREAK:
			ATGetDebugger()->Break();
			return true;
		case ID_DEBUG_RUN:
			ATGetDebugger()->Run();
			return true;
		case ID_DEBUG_TOGGLEBREAKPOINT:
			{
				int line = mpTextEditor->GetCursorLine();

				AddressLookup::const_iterator it(mLineToAddressLookup.find(line));
				if (it != mLineToAddressLookup.end())
					ATGetDebugger()->ToggleBreakpoint((uint16)it->second);
				else
					MessageBeep(MB_ICONEXCLAMATION);
			}
			return true;
		case ID_DEBUG_STEPINTO:
			ATGetDebugger()->StepInto();
			return true;
		case ID_DEBUG_STEPOVER:
			ATGetDebugger()->StepOver();
			return true;
		case ID_DEBUG_STEPOUT:
			ATGetDebugger()->StepOut();
			return true;

		case ID_CONTEXT_GOTODISASSEMBLY:
			{
				sint32 addr = GetCurrentLineAddress();

				if (addr >= 0) {
					ATActivateUIPane(kATUIPaneId_Disassembly, true);
					ATDisassemblyWindow *disPane = static_cast<ATDisassemblyWindow *>(ATGetUIPane(kATUIPaneId_Disassembly));
					if (disPane) {
						disPane->SetPosition((uint16)addr);
					}
				}
			}
			return true;

		case ID_CONTEXT_SETNEXTSTATEMENT:
			{
				sint32 addr = GetCurrentLineAddress();

				if (addr >= 0) {
					ATGetDebugger()->SetPC((uint16)addr);
				}
			}
			return true;
	}
	return false;
}

void ATSourceWindow::OnTextEditorUpdated() {
}

void ATSourceWindow::RecolorLine(int line, const char *text, int length, IVDTextEditorColorization *cl) {
	if (line == mHighlightedLine) {
		cl->AddTextColorPoint(0, 0x000000, 0xFFFF80);
		return;
	}

	AddressLookup::const_iterator it(mLineToAddressLookup.find(line));
	if (it != mLineToAddressLookup.end()) {
		uint32 addr = it->second;

		if (g_sim.GetCPU().IsBreakpointSet(addr)) {
			cl->AddTextColorPoint(0, 0x000000, 0xE0F0FF);
			return;
		}
	}

	for(int i=0; i<length; ++i) {
		char c = text[i];

		if (c == ';') {
			cl->AddTextColorPoint(0, 0x008000, -1);
			return;
		} else if (c == '.') {
			int j = i + 1;

			while(j < length) {
				c = text[j];

				if (!(c >= 'a' && c <= 'z') &&
					!(c >= 'A' && c <= 'Z') &&
					!(c >= '0' && c <= '9'))
				{
					break;
				}

				++j;
			}

			if (j > i+1) {
				cl->AddTextColorPoint(i, 0x0000FF, -1);
				cl->AddTextColorPoint(j, -1, -1);
			}

			return;
		}

		if (c != ' ' && c != '\t')
			break;
	}
}

void ATSourceWindow::OnSimulatorEvent(ATSimulatorEvent ev) {
	switch(ev) {
		case kATSimEvent_CPUPCBreakpointsUpdated:
			mpTextEditor->RecolorAll();
			break;
	}
}

void ATSourceWindow::OnDebuggerSystemStateUpdate(const ATDebuggerSystemState& state) {
	uint16 pc = g_sim.GetCPU().GetPC();

	AddressLookup::const_iterator it(mAddressToLineLookup.find(pc));
	if (it != mAddressToLineLookup.end()) {
		uint32 line = it->second;

		mpTextEditor->SetCursorPos(line, 0);
		HighlightLine(line);
	} else {
		HighlightLine(-1);
	}
}

bool ATSourceWindow::OnFileUpdated(const wchar_t *path) {
	if (mhwnd)
		PostMessage(mhwnd, WM_APP + 101, 0, 0);

	return true;
}

void ATSourceWindow::HighlightLine(int line) {
	if (line == mHighlightedLine)
		return;

	int oldLine = mHighlightedLine;
	mHighlightedLine = line;
	mpTextEditor->RecolorLine(oldLine);
	mpTextEditor->RecolorLine(line);
}

sint32 ATSourceWindow::GetCurrentLineAddress() const {
	int line = mpTextEditor->GetCursorLine();

	AddressLookup::const_iterator it(mLineToAddressLookup.find(line));

	if (it == mLineToAddressLookup.end())
		return -1;

	return it->second;
}

void ATLoadSourceFile(const wchar_t *s) {
	vdrefptr<ATSourceWindow> srcwin(new ATSourceWindow);

	HWND hwndSrcWin = (HWND)srcwin->Create(WS_EX_NOPARENTNOTIFY, WS_OVERLAPPEDWINDOW|WS_VISIBLE, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, (VDGUIHandle)g_hwnd, 0);

	if (hwndSrcWin) {
		try {
			srcwin->LoadFile(s);
		} catch(const MyError& e) {
			DestroyWindow(hwndSrcWin);
			e.post(g_hwnd, "Altirra error");
		}
	}
}

///////////////////////////////////////////////////////////////////////////

class ATConsoleWindow : public ATUIPane {
public:
	ATConsoleWindow();
	~ATConsoleWindow();

	void Activate() {
		if (mhwnd)
			SetForegroundWindow(mhwnd);
	}

	void Write(const char *s);

protected:
	LRESULT WndProc(UINT msg, WPARAM wParam, LPARAM lParam);

	bool OnCreate();
	void OnDestroy();
	void OnSize();
	LRESULT OnNotify(NMHDR *);

	HWND	mhwndLog;
	HWND	mhwndEdit;
	HMENU	mMenu;
};

ATConsoleWindow *g_pConsoleWindow;

ATConsoleWindow::ATConsoleWindow()
	: ATUIPane(kATUIPaneId_Console, "Console")
	, mMenu(LoadMenu(g_hInst, MAKEINTRESOURCE(IDR_DEBUGGER_MENU)))
{
	mPreferredDockCode = kATContainerDockBottom;
}

ATConsoleWindow::~ATConsoleWindow() {
	if (mMenu)
		DestroyMenu(mMenu);
}

LRESULT ATConsoleWindow::WndProc(UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
	case WM_SIZE:
		OnSize();
		return 0;
	case WM_NOTIFY:
		return OnNotify((NMHDR *)lParam);
	case WM_SETFOCUS:
		SetFocus(mhwndEdit);
		return 0;
	case WM_SYSCOMMAND:
		// block F10... unfortunately, this blocks plain Alt too
		if (!lParam)
			return 0;
		break;
	}

	return ATUIPane::WndProc(msg, wParam, lParam);
}

bool ATConsoleWindow::OnCreate() {
	if (!ATUIPane::OnCreate())
		return false;

	SetMenu(mhwnd, mMenu);

	mhwndLog = CreateWindowEx(WS_EX_CLIENTEDGE, "RICHEDIT", "", ES_READONLY|ES_MULTILINE|ES_AUTOVSCROLL|WS_VSCROLL|WS_VISIBLE|WS_CHILD, 0, 0, 0, 0, mhwnd, (HMENU)100, g_hInst, NULL);
	if (!mhwndLog)
		return false;

	SendMessage(mhwndLog, WM_SETFONT, (WPARAM)g_monoFont, NULL);

	mhwndEdit = CreateWindowEx(WS_EX_CLIENTEDGE, "RICHEDIT", "", WS_VISIBLE|WS_CHILD, 0, 0, 0, 0, mhwnd, (HMENU)100, g_hInst, NULL);
	if (!mhwndEdit)
		return false;

	SendMessage(mhwndEdit, EM_SETEVENTMASK, 0, ENM_KEYEVENTS);
	SendMessage(mhwndEdit, WM_SETFONT, (WPARAM)g_monoFont, NULL);

	OnSize();

	g_pConsoleWindow = this;
	return true;
}

void ATConsoleWindow::OnDestroy() {
	g_pConsoleWindow = NULL;

	SetMenu(mhwnd, NULL);

	ATUIPane::OnDestroy();
}

void ATConsoleWindow::OnSize() {
	RECT r;

	if (GetClientRect(mhwnd, &r)) {
		int h = 20;

		SetWindowPos(mhwndLog, NULL, 0, 0, r.right, r.bottom > h ? r.bottom - h : 0, SWP_NOMOVE|SWP_NOZORDER|SWP_NOACTIVATE|SWP_NOCOPYBITS);
		SetWindowPos(mhwndEdit, NULL, 0, r.bottom - h, r.right, h, SWP_NOZORDER|SWP_NOACTIVATE|SWP_NOCOPYBITS);
	}
}

LRESULT ATConsoleWindow::OnNotify(NMHDR *pHdr) {
	if (pHdr->hwndFrom == mhwndEdit) {
		if (pHdr->code == EN_MSGFILTER) {
			const MSGFILTER& mf = *(const MSGFILTER *)pHdr;

			if (mf.msg == WM_KEYDOWN || mf.msg == WM_SYSKEYDOWN) {
				if (mf.wParam == VK_CANCEL) {
					ATGetDebugger()->Break();
					return true;
				} else if (mf.wParam == VK_F11) {
					ATGetDebugger()->StepInto();
					return true;
				} else if (mf.wParam == VK_F10) {
					ATGetDebugger()->StepOver();
					return true;
				} else if (mf.wParam == VK_F5) {
					ATGetDebugger()->Run();
					return true;
				} else if (mf.wParam == VK_ESCAPE) {
					ATActivateUIPane(kATUIPaneId_Display, true);
					return true;
				}
			} else if (mf.msg == WM_KEYUP || mf.msg == WM_SYSKEYUP) {
				switch(mf.wParam) {
				case VK_CANCEL:
				case VK_F11:
				case VK_F10:
				case VK_F5:
				case VK_ESCAPE:
					return true;
				}
			} else if (mf.msg == WM_CHAR || mf.msg == WM_SYSCHAR) {
				if (mf.wParam == '\r') {
					int len = GetWindowTextLength(mhwndEdit);
					if (len) {
						char *buf = (char *)_alloca(len+1);
						buf[0] = 0;

						if (GetWindowText(mhwndEdit, buf, len+1)) {
							SetWindowText(mhwndEdit, "");
							ATConsoleExecuteCommand(buf);
						}
					}
					return true;
				}
			}

			return false;
		}
	}

	return 0;
}

void ATConsoleWindow::Write(const char *s) {
	if (SendMessage(mhwndLog, EM_GETLINECOUNT, 0, 0) > 5000) {
		POINT pt;
		SendMessage(mhwndLog, EM_GETSCROLLPOS, 0, (LPARAM)&pt);
		int idx = SendMessage(mhwndLog, EM_LINEINDEX, 2000, 0);
		SendMessage(mhwndLog, EM_SETSEL, 0, idx);
		SendMessage(mhwndLog, EM_REPLACESEL, FALSE, (LPARAM)"");
		SendMessage(mhwndLog, EM_SETSCROLLPOS, 0, (LPARAM)&pt);
	}

	char buf[2048];
	while(*s) {
		const char *eol = strchr(s, '\n');
		if (eol) {
			size_t len = eol - s;
			if (len < 2046) {
				memcpy(buf, s, len);
				buf[len] = '\r';
				buf[len+1] = '\n';
				buf[len+2] = 0;
				SendMessage(mhwndLog, EM_SETSEL, -1, -1);
				SendMessage(mhwndLog, EM_REPLACESEL, FALSE, (LPARAM)buf);
				s = eol+1;
				continue;
			}
		}

		SendMessage(mhwndLog, EM_SETSEL, -1, -1);
		SendMessage(mhwndLog, EM_REPLACESEL, FALSE, (LPARAM)s);
		break;
	}
}

///////////////////////////////////////////////////////////////////////////

void ATShowConsole() {
	ATActivateUIPane(kATUIPaneId_Console, !ATGetUIPane(kATUIPaneId_Console));
	ATActivateUIPane(kATUIPaneId_Registers, false);
}

void ATOpenConsole() {
	ATActivateUIPane(kATUIPaneId_Console, true);
	ATActivateUIPane(kATUIPaneId_Registers, false);
}

void ATCloseConsole() {
}

///////////////////////////////////////////////////////////////////////////

void ATInitUIPanes() {
	LoadLibrary("riched32");

	ATRegisterUIPaneType(kATUIPaneId_Registers, VDRefCountObjectFactory<ATRegistersWindow, ATUIPane>);
	ATRegisterUIPaneType(kATUIPaneId_Console, VDRefCountObjectFactory<ATConsoleWindow, ATUIPane>);
	ATRegisterUIPaneType(kATUIPaneId_Disassembly, VDRefCountObjectFactory<ATDisassemblyWindow, ATUIPane>);
	ATRegisterUIPaneType(kATUIPaneId_CallStack, VDRefCountObjectFactory<ATCallStackWindow, ATUIPane>);

	if (!g_monoFont) {
		g_monoFont = CreateFont(-10, 0, 0, 0, 0, FALSE, FALSE, FALSE, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_CHARACTER_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH | FF_DONTCARE, "Lucida Console");

		if (!g_monoFont)
			g_monoFont = (HFONT)GetStockObject(DEFAULT_GUI_FONT);
	}

	g_hmenuSrcContext = LoadMenu(g_hInst, MAKEINTRESOURCE(IDR_SOURCE_CONTEXT_MENU));
}

void ATShutdownUIPanes() {
	if (g_monoFont) {
		DeleteObject(g_monoFont);
		g_monoFont = NULL;
	}
}

void ATConsoleWrite(const char *s) {
	if (g_pConsoleWindow)
		g_pConsoleWindow->Write(s);
}

void ATConsolePrintf(const char *format, ...) {
	if (g_pConsoleWindow) {
		char buf[3072];
		va_list val;

		va_start(val, format);
		if ((unsigned)_vsnprintf(buf, 3072, format, val) < 3072)
			g_pConsoleWindow->Write(buf);
		va_end(val);
	}
}

void ATConsoleTaggedPrintf(const char *format, ...) {
	if (g_pConsoleWindow) {
		ATAnticEmulator& antic = g_sim.GetAntic();
		ATConsolePrintf("(%3d:%3d,%3d) ", antic.GetFrameCounter(), antic.GetBeamY(), antic.GetBeamX());

		char buf[3072];
		va_list val;

		va_start(val, format);
		if ((unsigned)_vsnprintf(buf, 3072, format, val) < 3072)
			g_pConsoleWindow->Write(buf);
		va_end(val);
	}
}
